# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Argument parser for `maastest.main`."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "prepare_parser",
    ]

import argparse

import distro_info
from maastest import utils
from six import text_type


class MAASTestArgumentParser(argparse.ArgumentParser):

    def parse_args(self, *args, **kwargs):
        args = super(MAASTestArgumentParser, self).parse_args(*args, **kwargs)
        bmc_details = (args.bmc_mac, args.bmc_username, args.bmc_password)
        if args.interactive:
            if bmc_details != (None, None, None):
                self.error(
                    "None of the BMC details should be provided when using "
                    "--interactive.")
        else:
            if args.bmc_mac is not None and args.bmc_ip is not None:
                self.error(
                    "BMC MAC address and IP address cannot be both provided")

            bmc_details_all_set = (
                (args.bmc_mac is not None or args.bmc_ip is not None) and
                args.bmc_username is not None and
                args.bmc_password is not None)
            if not bmc_details_all_set:
                self.error(
                    "All the BMC details (MAC address or IP address, username "
                    "and password) must be provided.")
        return args


latest_LTS_series = distro_info.UbuntuDistroInfo().lts()
latest_released_series = distro_info.UbuntuDistroInfo().stable()


def prepare_parser():
    # TODO: This is not tested sinc how maas-test will interface with
    # checkbox is still in flux.  Right now, this is just to get the
    # testing going.
    parser = MAASTestArgumentParser(
        description="Test whether or not a node is compatible with MAAS.")

    parser.add_argument(
        'interface', type=text_type,
        help="Network interface that connects the testing system to the "
             "dedicated testing network.  MAAS will serve DHCP on this "
             "interface.")
    parser.add_argument(
        '--archive', type=text_type, default=None,
        action='append',
        help="Additional personal package archive, 'sources.list' entry, or "
             "distribution component to install on the virtual MAAS system.  "
             "This can be specified multiple times.")
    parser.add_argument(
        '--interactive', action='store_true',
        help="Interactive mode.  Prompt the user to power up the node "
             "manually instead of doing it automatically through the BMC.")

    # BMC details.
    parser.add_argument(
        '--bmc-mac', type=text_type,
        help="MAC address of the node's baseboard management controller.  "
             "Use this if the BMC is connected to the interface given as "
             "argument. In non-interactive mode, either --bmc-mac or "
             "--bmc-ip.  Not needed in interactive mode.")
    parser.add_argument(
        '--bmc-ip', type=text_type,
        help="IP address of the node's baseboard management controller.  "
             "Use this if the BMC is not connected to the inteface given as "
             "argument.  Note that the IP address must not change for the "
             "duration of the testing.  In non-interactive mode, either "
             "--bmc-mac or --bmc-ip.  Not needed in interactive mode.")
    parser.add_argument(
        '--bmc-username', type=text_type,
        help="Username for authenticating to the node's BMC.  "
             "Not needed in interactive mode.")
    parser.add_argument(
        '--bmc-password', type=text_type,
        help="Password for authenticating to the node's BMC.  "
             "Not needed in interactive mode.")
    parser.add_argument(
        '--ipmi-driver', type=text_type, default='LAN_2_0',
        help="IPMI driver type for the node's BMC. Defaults to LAN_2_0 "
             "(IPMI v2.0). "
             "Other option is LAN (IPMI version 1.5).  Many BMCs can work "
             "with either IPMI 1.5 or 2.0, but, some BMCs such as HP's iLO "
             "only use IPMI 2.0. "
             "You may need to pass LAN if your BMC only supports IPMI 1.5.")

    # VM details.
    parser.add_argument(
        '--maas-series', type=text_type,
        choices=utils.get_supported_maas_series(),
        default=utils.get_supported_maas_series()[0],
        help="Code name for the Ubuntu release series to install on the "
             "virtual machine (where the MAAS server will be installed).  "
             "Defaults to the latest stable Ubuntu series (%(default)s).")
    parser.add_argument(
        '--maas-simplestreams-filter', type=text_type,
        default=None,
        help="Optional simplestreams filter used to retrieve the image for "
             "the virtual machine (e.g. 'label=beta2 other_filter=value'). "
             "maas-test will set the value for the 'arch' filter and for "
             "the 'series' filter (see option '--maas-series').  This is "
             "an advanced option used to specify additional simplestreams "
             "filter.")

    # MAAS images.
    parser.add_argument(
        '--series', type=text_type,
        choices=utils.get_supported_node_series(),
        # Hardcode 'trusty' until it's effectively the latest LTS.
        # This is the series used for commissioning and deployment and
        # MAAS doesn't support commissioning with precise.
        default='trusty',
        help="Code name for the Ubuntu release series that the node should "
             "run.  Defaults to the latest LTS (%(default)s).")
    parser.add_argument(
        '--simplestreams-filter', type=text_type,
        default=None,
        help="Optional simplestreams filter used to retrieve the ephemeral "
             "images MAAS downloads (e.g. 'label=beta2 other_filter=value'). "
             "MAAS will set the value for the 'arch' filter and for "
             "the 'series' filter (see option '--series').  This is "
             "an advanced option used to specify additional simplestreams "
             "filter.")
    parser.add_argument(
        '--architecture', type=text_type,
        default='amd64',
        help="Node's CPU architecture, e.g. armhf/highbank or amd64.  "
             "Defaults to %(default)s.")

    # Proxy.
    parser.add_argument(
        '--http-proxy', type=text_type,
        help="HTTP proxy to use for all downloads.  Implies --disable-cache.")
    parser.add_argument(
        '--disable-cache', action='store_true',
        help="By default, maas-test uses polipo to cache all KVM images, "
        "MAAS images and apt packages. If --disable-cache is specified, "
        "polipo will be disabled.")

    # Reporting.
    parser.add_argument(
        '--no-reporting', action='store_true',
        help="Log results only; do not store or submit them to Launchpad.")
    parser.add_argument(
        '--log-results-only', action='store_true',
        help="Store results, but do not submit them to Launchpad.")

    # Dry run
    parser.add_argument(
        '--dry-run', action='store_true',
        help="Don't attempt to boot any machines or do any destructive "
        "testing.")

    # VM Timeout
    parser.add_argument(
        # We use 300.0 as the default here because it's slightly longer
        # than uvtool's default 120 seconds. It's still not enough on
        # some virtual hardware, but we don't care too much about those.
        '--kvm-timeout', type=int, default=300.0,
        help="The number of seconds to wait for the maas-test virtual "
        "machine to come up.")

    return parser
