/* $Id$
 *
 * Abstract base class for all intermediate code nodes.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */


#ifndef __NODE_HPP_INCLUDED
#define __NODE_HPP_INCLUDED

#include "intermediate/visitor/Visitor.hpp"
#include "util/MiscUtil.hpp"
#include "util/GarbageCollect.hpp"
#include <list>

namespace intermediate {

//! base class for all visitible intermediate classes.
/** This class is an abstract base class for all intermediate
 *  code classes, that can get visited.
 */
class Node : public util::GarbageCollect {
public:

	//! c'tor
	Node() {}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor &visitor) = 0;

	//! shortcut type definition for integer annotations.
	typedef std::pair<std::string, int> intAnnoT;
	//! shortcut type definition for string annotations.
	typedef std::pair<std::string, std::string> strAnnoT;

	//! add an annotations in the form of name=int
	/** @param name name of the annotation
	 *  @param value integer value of the annotation
	 */
	void annotate(std::string name, int value) {
		this->intAnnotations.push_back(intAnnoT(name, value));
	}

	//! add an annotations in the form of name="string"
	/** @param name name of the annotation
	 *  @param value string value of the annotation
	 */
	void annotate(std::string name, std::string value) {
		this->strAnnotations.push_back(strAnnoT(name, value));
	}

	//! list of integer annotations
	std::list<intAnnoT> intAnnotations;
	//! list of string annotations
	std::list<strAnnoT> strAnnotations;

protected:
	/** virtual dummy d'tor. */
	virtual ~Node() {}
};

}; /* namespace intermediate */

#endif /* __NODE_HPP_INCLUDED */
