# search, a bzr plugin for searching within bzr branches/repositories.
# Copyright (C) 2008 Robert Collins
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""Tests for the commands supplied by search."""

from bzrlib.plugins.search.index import (
    index_url,
    init_index,
    open_index_url,
    )
from bzrlib.tests import TestCaseWithTransport


class TestSearch(TestCaseWithTransport):

    def test_no_parameters_error(self):
        self.run_bzr_error(['needs one or more'], ['search'])

    def test_no_index_error(self):
        self.run_bzr_error(['No search index'], ['search', 'robert'])

    def test_no_hits_error(self):
        branch = self.make_branch('.')
        init_index(branch)
        self.run_bzr_error(['No matches'], ['search', 'robert'])

    def test_simple_hits(self):
        tree = self.make_branch_and_tree('.')
        init_index(tree.branch)
        rev_id1 = tree.commit('first post')
        index_url(self.get_url('.'))
        index = open_index_url(self.get_url('.'))
        out, err = self.run_bzr(['search', 'post'])
        self.assertEqual('', err)
        self.assertEqual("Revision id '%s'. Summary: 'first post'\n" % rev_id1, out)

    def test_simple_exclusion(self):
        tree = self.make_branch_and_tree('.')
        init_index(tree.branch)
        rev_id1 = tree.commit('first post')
        rev_id2 = tree.commit('second post')
        index_url(self.get_url('.'))
        index = open_index_url(self.get_url('.'))
        out, err = self.run_bzr(['search', '--', 'post', '-first'])
        self.assertEqual('', err)
        self.assertEqual("Revision id '%s'. Summary: 'second post'\n" % rev_id2, out)

    def test_directory_option(self):
        tree = self.make_branch_and_tree('otherdir')
        init_index(tree.branch)
        rev_id1 = tree.commit('first post')
        index_url(self.get_url('otherdir'))
        out, err = self.run_bzr(['search', '-d', 'otherdir', 'post'])
        self.assertEqual('', err)
        self.assertEqual("Revision id '%s'. Summary: 'first post'\n" % rev_id1, out)

    def test_summary_first_line(self):
        tree = self.make_branch_and_tree('.')
        init_index(tree.branch)
        rev_id1 = tree.commit('this message\nhas two lines')
        index_url(self.get_url('.'))
        index = open_index_url(self.get_url('.'))
        out, err = self.run_bzr(['search', 'two'])
        self.assertEqual('', err)
        self.assertEqual("Revision id '%s'. Summary: 'this message'\n" % rev_id1, out)

    def test_search_suggestions_works(self):
        # Bare bones - no ui love as yet:
        tree = self.make_branch_and_tree('.')
        init_index(tree.branch)
        rev_id1 = tree.commit('this message\nhas two lines')
        rev_id2 = tree.commit('this message does not\n')
        index_url(self.get_url('.'))
        index = open_index_url(self.get_url('.'))
        out, err = self.run_bzr(['search', '-s', 'tw'])
        self.assertEqual('', err)
        self.assertEqual("Suggestions: [('two',)]\n", out)
        out, err = self.run_bzr(['search', '-s', 't'])
        self.assertEqual('', err)
        self.assertEqual("Suggestions: [('this',), ('two',)]\n", out)
        out, err = self.run_bzr(['search', '-s', 'too'])
        self.assertEqual('', err)
        self.assertEqual("Suggestions: []\n", out)


class TestIndex(TestCaseWithTransport):

    def test_index_branch(self):
        branch = self.make_branch('a-branch')
        out, error = self.run_bzr(['index', 'a-branch'])
        self.assertEqual('', error)
        self.assertEqual('', out)

    def test_index_branch_content(self):
        tree = self.make_branch_and_tree('a-branch')
        tree.commit('a search term')
        out, error = self.run_bzr(['index', 'a-branch'])
        self.assertEqual('', error)
        self.assertEqual('', out)
        self.assertSubset(set([('a',), ('search',), ('term',)]),
            set(dict(open_index_url('a-branch').all_terms())))

    def test_index_no_branch(self):
        self.run_bzr_error(['Not a branch'], ['index', '.'])

    def test_index_pwd_branch(self):
        tree = self.make_branch_and_tree('a-branch')
        out, error = self.run_bzr(['index'], working_dir='a-branch')
        self.assertEqual('', error)
        self.assertEqual('', out)
        open_index_url(self.get_url('a-branch'))
