#!/usr/bin/env python3
"""
Basic offspec example.
Sample is a grating on a substrate, modelled by a 1d lattice of long boxes
"""
import bornagain as ba
from bornagain import ba_plot as bp, deg, nm
import matplotlib.pyplot as plt


def get_sample():
    # Materials
    material_particle = ba.RefractiveMaterial("Particle", 0.0006, 2e-08)
    material_substrate = ba.RefractiveMaterial("Substrate", 6e-06, 2e-08)
    vacuum = ba.RefractiveMaterial("Vacuum", 0, 0)

    # Particles
    ff = ba.Box(1000*nm, 20*nm, 10*nm)
    particle = ba.Particle(material_particle, ff)
    particle_rotation = ba.RotationZ(90*deg)
    particle.rotate(particle_rotation)

    # Interference functions
    iff = ba.Interference1DLattice(100*nm, 0)
    iff_pdf = ba.Profile1DCauchy(1e6*nm)
    iff.setDecayFunction(iff_pdf)

    # Particle layouts
    layout = ba.ParticleLayout()
    layout.addParticle(particle)
    layout.setInterference(iff)
    layout.setTotalParticleSurfaceDensity(0.01)

    # Layers
    layer_1 = ba.Layer(vacuum)
    layer_1.addLayout(layout)
    layer_2 = ba.Layer(material_substrate)

    # Sample
    sample = ba.Sample()
    sample.addLayer(layer_1)
    sample.addLayer(layer_2)

    return sample


def get_simulation(sample):
    nla = 573
    naf = 200
    ny = 50
    scan = ba.LambdaScan(nla, 1*nm/nla, 1*nm)
    scan.setInclination(1.5 * deg)
    scan.setIntensity(1e9)
    detector = ba.OffspecDetector(ny, -1*deg, +1*deg, naf, 0.1*deg, 4*deg)
    return ba.OffspecSimulation(scan, sample, detector)


if __name__ == '__main__':
    sample = get_sample()
    simulation = get_simulation(sample)
    result = simulation.simulate()
    bp.plot_simulation_result(result)
    plt.show()
