//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Device/ScanEditor.cpp
//! @brief     Defines class ScanEditor.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Device/ScanEditor.h"
#include "Base/Axis/Frame.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Axis/PointwiseAxisItem.h"
#include "GUI/Model/Beam/BeamDistributionItem.h"
#include "GUI/Model/Beam/SourceItems.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/Model/Sim/InstrumentsSet.h"
#include "GUI/View/Device/AlphaScanEditor.h"
#include "GUI/View/Device/DistributionEditor.h"
#include "GUI/View/Device/FootprintForm.h"
#include "GUI/View/Numeric/DSpinBox.h"
#include <QFormLayout>
#include <QLineEdit>

ScanEditor::ScanEditor(QWidget* parent, InstrumentItem* instr_item, ScanItem* item,
                       bool allow_footprint, bool allow_distr)
    : CollapsibleGroupBox("Beam and scan parameters", parent, item->expandBeamParameters)
{
    auto* layout = new QGridLayout(body());
    layout->setAlignment(Qt::AlignTop);

    //... Intensity

    auto* form = new QFormLayout;
    layout->addLayout(form, 0, 0);
    form->setFieldGrowthPolicy(QFormLayout::FieldsStayAtSizeHint);
    auto* intensity_spinbox = new DSpinBox(&item->intensity());
    form->addRow("Intensity:", intensity_spinbox);
    intensity_spinbox->setToolTip("Correction factor for normalized intensity");

    //... Scan unit (deg vs 1/nm)
    // TODO

    //... Wavelength

    auto* wavelengthEditor = new DistributionEditor("Wavelength (nm)", MeanConfig{true},
                                                    DistributionSelector::Category::Symmetric, this,
                                                    item->wavelengthItem(), allow_distr);
    layout->addWidget(wavelengthEditor, 1, 0);

    //... Inclination scan (alpha or qz)

    auto* inclinationEditor = new AlphaScanEditor(this, item, allow_distr);
    layout->addWidget(inclinationEditor, 1, 1);

    //... Signals, will couple wavelength and inclination

    // In the following, signal order is important! Indicators have to be recalculated first,
    // then updated (recalculation is done in EditController)
    connect(wavelengthEditor, &DistributionEditor::distributionChanged,
            [instr_item, inclinationEditor] {
                // Update values in pointwise axis. This is necessary after a change of the mean
                // wavelength.
                if (auto* iiI = dynamic_cast<SpecularInstrumentItem*>(instr_item)) {
                    if (auto* pw = dynamic_cast<PointwiseAxisItem*>(
                            iiI->scanItem()->inclinationAxisItem())) {
                        const Frame frame = instr_item->makeFrame();
                        pw->updateAxIndicators(frame);
                    }
                }
                emit gDoc->instruments()->currentModified();
                emit inclinationEditor->updateIndicators();
            });

    connect(inclinationEditor, &AlphaScanEditor::dataChanged, [wavelengthEditor] {
        emit gDoc->instruments()->currentModified();
        emit wavelengthEditor->updateData();
    });

    //... Footprint

    if (allow_footprint) {
        auto* footprintEditor = new FootprintForm(this, item);
        layout->addWidget(footprintEditor, 1, 2);
        connect(footprintEditor, &FootprintForm::dataChanged, this, &ScanEditor::dataChanged);
    }
}
