// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/RegComp.h"
#include "Puma/ErrorStream.h"

#include "lexertl/rules.hpp"
#include "lexertl/generator.hpp"
#include "lexertl/lookup.hpp"

using namespace std;

namespace Puma {


// Constructors
RegComp::RegComp (const std::string &expr) : _expr (expr), _compiled (false) {}

RegComp::RegComp (const RegComp &rc) : _expr (rc._expr), _compiled (false) {
}

RegComp &RegComp::operator = (const RegComp &rc) {
  _expr = rc._expr;
  _error_msg.clear ();
  _compiled = false;
  _sm.clear();
  return *this;
}

// Compile the regular expression.
bool RegComp::compile (const string &expr) {
  _compiled = true;
  lexertl::rules rules;
  rules.add (_expr, 1); // id is '1', because it is not needed; but != 0
  try {
    lexertl::generator::build(rules, _sm); // gen. state machine representation
  }
  catch (const std::exception &e) {
    _error_msg = e.what ();
    return false;
  }
  return true;
}


// Match against a string.
bool RegComp::match (const string &str /*, int eflags*/) {
  if (!_compiled)
    compile (_expr);
  string input (str);
  string::const_iterator iter = input.begin ();
  string::const_iterator end = input.end ();
  lexertl::match_results results (iter, end);
  try {
    lexertl::lookup (_sm, results);
  }
  catch (const std::exception &e) {
    _error_msg = e.what ();
    _error_msg += string ("; matched '") + str + "' with regex '" + _expr + "'";
    return false;
  }

  return (results.id == 1);
}
	

// To be called on errors.
void RegComp::comperror (ErrorStream &err) const {
  if (error ())
    err << sev_error << "RegComp: " << _error_msg.c_str () << endMessage;
}


} // namespace Puma
