/*
 * Copyright (c) 2019-2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_hybrid_fp32_mla_8x4 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const float *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const float *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x8\n"
      "bge 148f\n"
      "cmp %x[M], #0x6\n"
      "bgt 127f\n"
      "beq 106f\n"
      "cmp %x[M], #0x4\n"
      "bgt 85f\n"
      "beq 64f\n"
      "cmp %x[M], #0x2\n"
      "bgt 43f\n"
      "beq 22f\n"
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "cbz x14, 3f\n"
      "ldr q24, [x14, #0x0]\n"
      "add x14, x14, #0x10\n"
      "b 8f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 7f\n"
      "cmp x13, #0x4\n"
      "bge 6f\n"
      "tbz x13, #1, 4f\n"
      "ldr d24, [x11], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x13, #0, 5f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "b 5f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "mov x20, #0x0\n"
      "5:"  // Height 1: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 8f\n"
      "6:"  // Height 1: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "b 8f\n"
      "7:"  // Height 1: no accumulate
      "movi v24.16b, #0x0\n"
      "8:"  // Height 1: setup done
      "mov x10, #0x0\n"
      "9:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 10f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "cbnz x10, 11f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "b 11f\n"
      "10:"  // Height 1: setup direct input
      "mov x28, %x[input_ptr]\n"
      "11:"  // Height 1: input setup done
      "cmp x9, #0x4\n"
      "blt 14f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 13f\n"
      "12:"  // Height 1: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "cmp x9, #0x8\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "bge 12b\n"
      "13:"  // Height 1: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x28, x28, #0x10\n"
      "sub x9, x9, #0x4\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "add x12, x12, #0x40\n"
      "14:"  // Height 1: Multiply loop: Main loop skip
      "cbz x9, 16f\n"
      "15:"  // Height 1: Multiply loop: Odd block loop
      "ldr s17, [x28], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "sub x9, x9, #0x1\n"
      "fmla v24.4s, v16.4s, v17.s[0]\n"
      "add x12, x12, #0x10\n"
      "cbnz x9, 15b\n"
      "16:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 9b\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "tbz %x[flags], #1, 17f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "17:"  // Height 1: No activation
      "cmp x13, #0x4\n"
      "bge 20f\n"
      "tbz x13, #1, 18f\n"
      "str d24, [x11], #0x8\n"
      "tbz x13, #0, 19f\n"
      "st1 { v24.s }[2], [x11]\n"
      "b 19f\n"
      "18:"  // Height 1: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "19:"  // Height 1: Partial direct writeback: Done
      "b 21f\n"
      "20:"  // Height 1: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "21:"  // Height 1: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 2b\n"
      "b 170f\n"
      "22:"  // Height 2
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "23:"  // Height 2: Column loop
      "cbz x14, 24f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "b 29f\n"
      "24:"  // Height 2: no bias
      "tbz %x[flags], #0, 28f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x13, #0x4\n"
      "add x27, x11, x20, LSL #2\n"
      "bge 27f\n"
      "tbz x13, #1, 25f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x13, #0, 26f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "b 26f\n"
      "25:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "26:"  // Height 2: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 29f\n"
      "27:"  // Height 2: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "b 29f\n"
      "28:"  // Height 2: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "29:"  // Height 2: setup done
      "mov x10, #0x0\n"
      "30:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "cbnz x10, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "b 32f\n"
      "31:"  // Height 2: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "32:"  // Height 2: input setup done
      "cmp x9, #0x4\n"
      "blt 35f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 34f\n"
      "33:"  // Height 2: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "add x27, x27, #0x10\n"
      "cmp x9, #0x8\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "bge 33b\n"
      "34:"  // Height 2: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "sub x9, x9, #0x4\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "35:"  // Height 2: Multiply loop: Main loop skip
      "cbz x9, 37f\n"
      "36:"  // Height 2: Multiply loop: Odd block loop
      "ldr s18, [x28], #0x4\n"
      "ldr s17, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v18.s[0]\n"
      "fmla v25.4s, v16.4s, v17.s[0]\n"
      "add x12, x12, #0x10\n"
      "cbnz x9, 36b\n"
      "37:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 30b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "tbz %x[flags], #1, 38f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "38:"  // Height 2: No activation
      "cmp x13, #0x4\n"
      "bge 41f\n"
      "tbz x13, #1, 39f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "tbz x13, #0, 40f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "b 40f\n"
      "39:"  // Height 2: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "40:"  // Height 2: Partial direct writeback: Done
      "b 42f\n"
      "41:"  // Height 2: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "42:"  // Height 2: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 23b\n"
      "b 170f\n"
      "43:"  // Height 3
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "44:"  // Height 3: Column loop
      "cbz x14, 45f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "b 50f\n"
      "45:"  // Height 3: no bias
      "tbz %x[flags], #0, 49f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x26, x27, x20, LSL #2\n"
      "bge 48f\n"
      "tbz x13, #1, 46f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "tbz x13, #0, 47f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "b 47f\n"
      "46:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "47:"  // Height 3: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 50f\n"
      "48:"  // Height 3: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "b 50f\n"
      "49:"  // Height 3: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "50:"  // Height 3: setup done
      "mov x10, #0x0\n"
      "51:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 52f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "cbnz x10, 53f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "b 53f\n"
      "52:"  // Height 3: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "53:"  // Height 3: input setup done
      "cmp x9, #0x4\n"
      "blt 56f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 55f\n"
      "54:"  // Height 3: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "cmp x9, #0x8\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "bge 54b\n"
      "55:"  // Height 3: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x26, x26, #0x10\n"
      "sub x9, x9, #0x4\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "56:"  // Height 3: Multiply loop: Main loop skip
      "cbz x9, 58f\n"
      "57:"  // Height 3: Multiply loop: Odd block loop
      "ldr s19, [x28], #0x4\n"
      "ldr s18, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s17, [x26], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v19.s[0]\n"
      "fmla v25.4s, v16.4s, v18.s[0]\n"
      "fmla v26.4s, v16.4s, v17.s[0]\n"
      "add x12, x12, #0x10\n"
      "cbnz x9, 57b\n"
      "58:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 51b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "tbz %x[flags], #1, 59f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "59:"  // Height 3: No activation
      "cmp x13, #0x4\n"
      "bge 62f\n"
      "tbz x13, #1, 60f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "tbz x13, #0, 61f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "b 61f\n"
      "60:"  // Height 3: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "61:"  // Height 3: Partial direct writeback: Done
      "b 63f\n"
      "62:"  // Height 3: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "63:"  // Height 3: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 44b\n"
      "b 170f\n"
      "64:"  // Height 4
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "65:"  // Height 4: Column loop
      "cbz x14, 66f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "mov v27.16b, v24.16b\n"
      "b 71f\n"
      "66:"  // Height 4: no bias
      "tbz %x[flags], #0, 70f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x25, x26, x20, LSL #2\n"
      "bge 69f\n"
      "tbz x13, #1, 67f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "ldr d27, [x25], #0x8\n"
      "tbz x13, #0, 68f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "ld1 { v27.s }[2], [x25]\n"
      "b 68f\n"
      "67:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "ldr s27, [x25, #0x0]\n"
      "68:"  // Height 4: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 71f\n"
      "69:"  // Height 4: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "ldr q27, [x25, #0x0]\n"
      "b 71f\n"
      "70:"  // Height 4: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "71:"  // Height 4: setup done
      "mov x10, #0x0\n"
      "72:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 73f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "cbnz x10, 74f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "b 74f\n"
      "73:"  // Height 4: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "74:"  // Height 4: input setup done
      "cmp x9, #0x4\n"
      "blt 77f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 76f\n"
      "75:"  // Height 4: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "add x25, x25, #0x10\n"
      "cmp x9, #0x8\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "bge 75b\n"
      "76:"  // Height 4: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "sub x9, x9, #0x4\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "77:"  // Height 4: Multiply loop: Main loop skip
      "cbz x9, 79f\n"
      "78:"  // Height 4: Multiply loop: Odd block loop
      "ldr s20, [x28], #0x4\n"
      "ldr s19, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s18, [x26], #0x4\n"
      "ldr s17, [x25], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v20.s[0]\n"
      "fmla v25.4s, v16.4s, v19.s[0]\n"
      "add x12, x12, #0x10\n"
      "fmla v26.4s, v16.4s, v18.s[0]\n"
      "fmla v27.4s, v16.4s, v17.s[0]\n"
      "cbnz x9, 78b\n"
      "79:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 72b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x25, x26, x20, LSL #2\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "tbz %x[flags], #1, 80f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmin v27.4s, v27.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "fmax v27.4s, v27.4s, v16.4s\n"
      "80:"  // Height 4: No activation
      "cmp x13, #0x4\n"
      "bge 83f\n"
      "tbz x13, #1, 81f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "str d27, [x25], #0x8\n"
      "tbz x13, #0, 82f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "st1 { v27.s }[2], [x25]\n"
      "b 82f\n"
      "81:"  // Height 4: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "str s27, [x25, #0x0]\n"
      "82:"  // Height 4: Partial direct writeback: Done
      "b 84f\n"
      "83:"  // Height 4: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "str q27, [x25, #0x0]\n"
      "84:"  // Height 4: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 65b\n"
      "b 170f\n"
      "85:"  // Height 5
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "86:"  // Height 5: Column loop
      "cbz x14, 87f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "mov v27.16b, v24.16b\n"
      "mov v28.16b, v24.16b\n"
      "b 92f\n"
      "87:"  // Height 5: no bias
      "tbz %x[flags], #0, 91f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x24, x25, x20, LSL #2\n"
      "bge 90f\n"
      "tbz x13, #1, 88f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "ldr d27, [x25], #0x8\n"
      "ldr d28, [x24], #0x8\n"
      "tbz x13, #0, 89f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "ld1 { v27.s }[2], [x25]\n"
      "ld1 { v28.s }[2], [x24]\n"
      "b 89f\n"
      "88:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "ldr s27, [x25, #0x0]\n"
      "ldr s28, [x24, #0x0]\n"
      "89:"  // Height 5: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 92f\n"
      "90:"  // Height 5: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "ldr q27, [x25, #0x0]\n"
      "ldr q28, [x24, #0x0]\n"
      "b 92f\n"
      "91:"  // Height 5: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "92:"  // Height 5: setup done
      "mov x10, #0x0\n"
      "93:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 94f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "cbnz x10, 95f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "b 95f\n"
      "94:"  // Height 5: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "95:"  // Height 5: input setup done
      "cmp x9, #0x4\n"
      "blt 98f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q4, [x24, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 97f\n"
      "96:"  // Height 5: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "cmp x9, #0x8\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "ldr q9, [x12, #0x10]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "ldr q3, [x25, #0x0]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "ldr q4, [x24, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "bge 96b\n"
      "97:"  // Height 5: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x24, x24, #0x10\n"
      "sub x9, x9, #0x4\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "98:"  // Height 5: Multiply loop: Main loop skip
      "cbz x9, 100f\n"
      "99:"  // Height 5: Multiply loop: Odd block loop
      "ldr s21, [x28], #0x4\n"
      "ldr s20, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s19, [x26], #0x4\n"
      "ldr s18, [x25], #0x4\n"
      "ldr s17, [x24], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v21.s[0]\n"
      "fmla v25.4s, v16.4s, v20.s[0]\n"
      "fmla v26.4s, v16.4s, v19.s[0]\n"
      "fmla v27.4s, v16.4s, v18.s[0]\n"
      "add x12, x12, #0x10\n"
      "fmla v28.4s, v16.4s, v17.s[0]\n"
      "cbnz x9, 99b\n"
      "100:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 93b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "tbz %x[flags], #1, 101f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmin v27.4s, v27.4s, v17.4s\n"
      "fmin v28.4s, v28.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "fmax v27.4s, v27.4s, v16.4s\n"
      "fmax v28.4s, v28.4s, v16.4s\n"
      "101:"  // Height 5: No activation
      "cmp x13, #0x4\n"
      "bge 104f\n"
      "tbz x13, #1, 102f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "str d27, [x25], #0x8\n"
      "str d28, [x24], #0x8\n"
      "tbz x13, #0, 103f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "st1 { v27.s }[2], [x25]\n"
      "st1 { v28.s }[2], [x24]\n"
      "b 103f\n"
      "102:"  // Height 5: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "str s27, [x25, #0x0]\n"
      "str s28, [x24, #0x0]\n"
      "103:"  // Height 5: Partial direct writeback: Done
      "b 105f\n"
      "104:"  // Height 5: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "str q27, [x25, #0x0]\n"
      "str q28, [x24, #0x0]\n"
      "105:"  // Height 5: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 86b\n"
      "b 170f\n"
      "106:"  // Height 6
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "107:"  // Height 6: Column loop
      "cbz x14, 108f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "mov v27.16b, v24.16b\n"
      "mov v28.16b, v24.16b\n"
      "mov v29.16b, v24.16b\n"
      "b 113f\n"
      "108:"  // Height 6: no bias
      "tbz %x[flags], #0, 112f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 111f\n"
      "tbz x13, #1, 109f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "ldr d27, [x25], #0x8\n"
      "ldr d28, [x24], #0x8\n"
      "ldr d29, [x23], #0x8\n"
      "tbz x13, #0, 110f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "ld1 { v27.s }[2], [x25]\n"
      "ld1 { v28.s }[2], [x24]\n"
      "ld1 { v29.s }[2], [x23]\n"
      "b 110f\n"
      "109:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "ldr s27, [x25, #0x0]\n"
      "ldr s28, [x24, #0x0]\n"
      "ldr s29, [x23, #0x0]\n"
      "110:"  // Height 6: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 113f\n"
      "111:"  // Height 6: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "ldr q27, [x25, #0x0]\n"
      "ldr q28, [x24, #0x0]\n"
      "ldr q29, [x23, #0x0]\n"
      "b 113f\n"
      "112:"  // Height 6: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "113:"  // Height 6: setup done
      "mov x10, #0x0\n"
      "114:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 115f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "cbnz x10, 116f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 116f\n"
      "115:"  // Height 6: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "116:"  // Height 6: input setup done
      "cmp x9, #0x4\n"
      "blt 119f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q4, [x24, #0x0]\n"
      "ldr q5, [x23, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 118f\n"
      "117:"  // Height 6: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "add x23, x23, #0x10\n"
      "cmp x9, #0x8\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "ldr q9, [x12, #0x10]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "ldr q3, [x25, #0x0]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "ldr q4, [x24, #0x0]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "ldr q5, [x23, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "bge 117b\n"
      "118:"  // Height 6: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "sub x9, x9, #0x4\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "119:"  // Height 6: Multiply loop: Main loop skip
      "cbz x9, 121f\n"
      "120:"  // Height 6: Multiply loop: Odd block loop
      "ldr s22, [x28], #0x4\n"
      "ldr s21, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s20, [x26], #0x4\n"
      "ldr s19, [x25], #0x4\n"
      "ldr s18, [x24], #0x4\n"
      "ldr s17, [x23], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v22.s[0]\n"
      "fmla v25.4s, v16.4s, v21.s[0]\n"
      "add x12, x12, #0x10\n"
      "fmla v26.4s, v16.4s, v20.s[0]\n"
      "fmla v27.4s, v16.4s, v19.s[0]\n"
      "fmla v28.4s, v16.4s, v18.s[0]\n"
      "fmla v29.4s, v16.4s, v17.s[0]\n"
      "cbnz x9, 120b\n"
      "121:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 114b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "tbz %x[flags], #1, 122f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmin v27.4s, v27.4s, v17.4s\n"
      "fmin v28.4s, v28.4s, v17.4s\n"
      "fmin v29.4s, v29.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "fmax v27.4s, v27.4s, v16.4s\n"
      "fmax v28.4s, v28.4s, v16.4s\n"
      "fmax v29.4s, v29.4s, v16.4s\n"
      "122:"  // Height 6: No activation
      "cmp x13, #0x4\n"
      "bge 125f\n"
      "tbz x13, #1, 123f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "str d27, [x25], #0x8\n"
      "str d28, [x24], #0x8\n"
      "str d29, [x23], #0x8\n"
      "tbz x13, #0, 124f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "st1 { v27.s }[2], [x25]\n"
      "st1 { v28.s }[2], [x24]\n"
      "st1 { v29.s }[2], [x23]\n"
      "b 124f\n"
      "123:"  // Height 6: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "str s27, [x25, #0x0]\n"
      "str s28, [x24, #0x0]\n"
      "str s29, [x23, #0x0]\n"
      "124:"  // Height 6: Partial direct writeback: Done
      "b 126f\n"
      "125:"  // Height 6: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "str q27, [x25, #0x0]\n"
      "str q28, [x24, #0x0]\n"
      "str q29, [x23, #0x0]\n"
      "126:"  // Height 6: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 107b\n"
      "b 170f\n"
      "127:"  // Height 7
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "128:"  // Height 7: Column loop
      "cbz x14, 129f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "mov v27.16b, v24.16b\n"
      "mov v28.16b, v24.16b\n"
      "mov v29.16b, v24.16b\n"
      "mov v30.16b, v24.16b\n"
      "b 134f\n"
      "129:"  // Height 7: no bias
      "tbz %x[flags], #0, 133f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 132f\n"
      "tbz x13, #1, 130f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "ldr d27, [x25], #0x8\n"
      "ldr d28, [x24], #0x8\n"
      "ldr d29, [x23], #0x8\n"
      "ldr d30, [x22], #0x8\n"
      "tbz x13, #0, 131f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "ld1 { v27.s }[2], [x25]\n"
      "ld1 { v28.s }[2], [x24]\n"
      "ld1 { v29.s }[2], [x23]\n"
      "ld1 { v30.s }[2], [x22]\n"
      "b 131f\n"
      "130:"  // Height 7: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "ldr s27, [x25, #0x0]\n"
      "ldr s28, [x24, #0x0]\n"
      "ldr s29, [x23, #0x0]\n"
      "ldr s30, [x22, #0x0]\n"
      "131:"  // Height 7: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 134f\n"
      "132:"  // Height 7: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "ldr q27, [x25, #0x0]\n"
      "ldr q28, [x24, #0x0]\n"
      "ldr q29, [x23, #0x0]\n"
      "ldr q30, [x22, #0x0]\n"
      "b 134f\n"
      "133:"  // Height 7: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "134:"  // Height 7: setup done
      "mov x10, #0x0\n"
      "135:"  // Height 7: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 136f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "cbnz x10, 137f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 137f\n"
      "136:"  // Height 7: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "137:"  // Height 7: input setup done
      "cmp x9, #0x4\n"
      "blt 140f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q4, [x24, #0x0]\n"
      "ldr q5, [x23, #0x0]\n"
      "ldr q6, [x22, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 139f\n"
      "138:"  // Height 7: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v30.4s, v8.4s, v6.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "cmp x9, #0x8\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "fmla v30.4s, v9.4s, v6.s[1]\n"
      "ldr q9, [x12, #0x10]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "fmla v30.4s, v10.4s, v6.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "ldr q3, [x25, #0x0]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "ldr q4, [x24, #0x0]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "ldr q5, [x23, #0x0]\n"
      "fmla v30.4s, v11.4s, v6.s[3]\n"
      "ldr q6, [x22, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "bge 138b\n"
      "139:"  // Height 7: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla v30.4s, v8.4s, v6.s[0]\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "add x22, x22, #0x10\n"
      "sub x9, x9, #0x4\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "fmla v30.4s, v9.4s, v6.s[1]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "fmla v30.4s, v10.4s, v6.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "fmla v30.4s, v11.4s, v6.s[3]\n"
      "140:"  // Height 7: Multiply loop: Main loop skip
      "cbz x9, 142f\n"
      "141:"  // Height 7: Multiply loop: Odd block loop
      "ldr s23, [x28], #0x4\n"
      "ldr s22, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s21, [x26], #0x4\n"
      "ldr s20, [x25], #0x4\n"
      "ldr s19, [x24], #0x4\n"
      "ldr s18, [x23], #0x4\n"
      "ldr s17, [x22], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v23.s[0]\n"
      "fmla v25.4s, v16.4s, v22.s[0]\n"
      "fmla v26.4s, v16.4s, v21.s[0]\n"
      "fmla v27.4s, v16.4s, v20.s[0]\n"
      "add x12, x12, #0x10\n"
      "fmla v28.4s, v16.4s, v19.s[0]\n"
      "fmla v29.4s, v16.4s, v18.s[0]\n"
      "fmla v30.4s, v16.4s, v17.s[0]\n"
      "cbnz x9, 141b\n"
      "142:"  // Height 7: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 135b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "tbz %x[flags], #1, 143f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmin v27.4s, v27.4s, v17.4s\n"
      "fmin v28.4s, v28.4s, v17.4s\n"
      "fmin v29.4s, v29.4s, v17.4s\n"
      "fmin v30.4s, v30.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "fmax v27.4s, v27.4s, v16.4s\n"
      "fmax v28.4s, v28.4s, v16.4s\n"
      "fmax v29.4s, v29.4s, v16.4s\n"
      "fmax v30.4s, v30.4s, v16.4s\n"
      "143:"  // Height 7: No activation
      "cmp x13, #0x4\n"
      "bge 146f\n"
      "tbz x13, #1, 144f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "str d27, [x25], #0x8\n"
      "str d28, [x24], #0x8\n"
      "str d29, [x23], #0x8\n"
      "str d30, [x22], #0x8\n"
      "tbz x13, #0, 145f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "st1 { v27.s }[2], [x25]\n"
      "st1 { v28.s }[2], [x24]\n"
      "st1 { v29.s }[2], [x23]\n"
      "st1 { v30.s }[2], [x22]\n"
      "b 145f\n"
      "144:"  // Height 7: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "str s27, [x25, #0x0]\n"
      "str s28, [x24, #0x0]\n"
      "str s29, [x23, #0x0]\n"
      "str s30, [x22, #0x0]\n"
      "145:"  // Height 7: Partial direct writeback: Done
      "b 147f\n"
      "146:"  // Height 7: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "str q27, [x25, #0x0]\n"
      "str q28, [x24, #0x0]\n"
      "str q29, [x23, #0x0]\n"
      "str q30, [x22, #0x0]\n"
      "147:"  // Height 7: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 128b\n"
      "b 170f\n"
      "148:"  // Height 8
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x20\n"
      "mov x14, %x[bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x11, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "149:"  // Height 8: Column loop
      "cbz x14, 150f\n"
      "ldr q24, [x14, #0x0]\n"
      "mov v25.16b, v24.16b\n"
      "mov v26.16b, v24.16b\n"
      "add x14, x14, #0x10\n"
      "mov v27.16b, v24.16b\n"
      "mov v28.16b, v24.16b\n"
      "mov v29.16b, v24.16b\n"
      "mov v30.16b, v24.16b\n"
      "mov v31.16b, v24.16b\n"
      "b 155f\n"
      "150:"  // Height 8: no bias
      "tbz %x[flags], #0, 154f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "cmp x13, #0x4\n"
      "add x21, x22, x20, LSL #2\n"
      "bge 153f\n"
      "tbz x13, #1, 151f\n"
      "ldr d24, [x11], #0x8\n"
      "ldr d25, [x27], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d26, [x26], #0x8\n"
      "ldr d27, [x25], #0x8\n"
      "ldr d28, [x24], #0x8\n"
      "ldr d29, [x23], #0x8\n"
      "ldr d30, [x22], #0x8\n"
      "ldr d31, [x21], #0x8\n"
      "tbz x13, #0, 152f\n"
      "ld1 { v24.s }[2], [x11]\n"
      "ld1 { v25.s }[2], [x27]\n"
      "ld1 { v26.s }[2], [x26]\n"
      "ld1 { v27.s }[2], [x25]\n"
      "ld1 { v28.s }[2], [x24]\n"
      "ld1 { v29.s }[2], [x23]\n"
      "ld1 { v30.s }[2], [x22]\n"
      "ld1 { v31.s }[2], [x21]\n"
      "b 152f\n"
      "151:"  // Height 8: Partial accumulate: partial_1_0
      "ldr s24, [x11, #0x0]\n"
      "ldr s25, [x27, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s26, [x26, #0x0]\n"
      "ldr s27, [x25, #0x0]\n"
      "ldr s28, [x24, #0x0]\n"
      "ldr s29, [x23, #0x0]\n"
      "ldr s30, [x22, #0x0]\n"
      "ldr s31, [x21, #0x0]\n"
      "152:"  // Height 8: Partial accumulate: Done
      "sub x11, x11, x20\n"
      "b 155f\n"
      "153:"  // Height 8: full accumulate
      "ldr q24, [x11, #0x0]\n"
      "ldr q25, [x27, #0x0]\n"
      "ldr q26, [x26, #0x0]\n"
      "ldr q27, [x25, #0x0]\n"
      "ldr q28, [x24, #0x0]\n"
      "ldr q29, [x23, #0x0]\n"
      "ldr q30, [x22, #0x0]\n"
      "ldr q31, [x21, #0x0]\n"
      "b 155f\n"
      "154:"  // Height 8: no accumulate
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "155:"  // Height 8: setup done
      "mov x10, #0x0\n"
      "156:"  // Height 8: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 157f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "ldr x21, [x20, #0x38]\n"
      "cbnz x10, 158f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 158f\n"
      "157:"  // Height 8: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "158:"  // Height 8: input setup done
      "cmp x9, #0x4\n"
      "blt 161f\n"
      "ldr q0, [x28, #0x0]\n"
      "ldr q1, [x27, #0x0]\n"
      "cmp x9, #0x8\n"
      "ldr q2, [x26, #0x0]\n"
      "ldr q3, [x25, #0x0]\n"
      "ldr q4, [x24, #0x0]\n"
      "ldr q5, [x23, #0x0]\n"
      "ldr q6, [x22, #0x0]\n"
      "ldr q7, [x21, #0x0]\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "blt 160f\n"
      "159:"  // Height 8: Multiply loop: Main loop head
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "sub x9, x9, #0x4\n"
      "add x28, x28, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x27, x27, #0x10\n"
      "add x26, x26, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      "fmla v30.4s, v8.4s, v6.s[0]\n"
      "fmla v31.4s, v8.4s, v7.s[0]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "add x21, x21, #0x10\n"
      "cmp x9, #0x8\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "add x12, x12, #0x40\n"
      "ldr q8, [x12, #0x0]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "fmla v30.4s, v9.4s, v6.s[1]\n"
      "fmla v31.4s, v9.4s, v7.s[1]\n"
      "ldr q9, [x12, #0x10]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "fmla v30.4s, v10.4s, v6.s[2]\n"
      "fmla v31.4s, v10.4s, v7.s[2]\n"
      "ldr q10, [x12, #0x20]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "ldr q0, [x28, #0x0]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "ldr q1, [x27, #0x0]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "ldr q2, [x26, #0x0]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "ldr q3, [x25, #0x0]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "ldr q4, [x24, #0x0]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "ldr q5, [x23, #0x0]\n"
      "fmla v30.4s, v11.4s, v6.s[3]\n"
      "ldr q6, [x22, #0x0]\n"
      "fmla v31.4s, v11.4s, v7.s[3]\n"
      "ldr q7, [x21, #0x0]\n"
      "ldr q11, [x12, #0x30]\n"
      "bge 159b\n"
      "160:"  // Height 8: Multiply loop: Single iteration only
      "fmla v24.4s, v8.4s, v0.s[0]\n"
      "fmla v25.4s, v8.4s, v1.s[0]\n"
      "add x28, x28, #0x10\n"
      "add x27, x27, #0x10\n"
      "fmla v26.4s, v8.4s, v2.s[0]\n"
      "fmla v27.4s, v8.4s, v3.s[0]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla v28.4s, v8.4s, v4.s[0]\n"
      "fmla v29.4s, v8.4s, v5.s[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla v30.4s, v8.4s, v6.s[0]\n"
      "fmla v31.4s, v8.4s, v7.s[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "fmla v24.4s, v9.4s, v0.s[1]\n"
      "fmla v25.4s, v9.4s, v1.s[1]\n"
      "sub x9, x9, #0x4\n"
      "prfm pldl1keep, [x28, #0x80]\n"
      "fmla v26.4s, v9.4s, v2.s[1]\n"
      "fmla v27.4s, v9.4s, v3.s[1]\n"
      "prfm pldl1keep, [x27, #0x80]\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "fmla v28.4s, v9.4s, v4.s[1]\n"
      "fmla v29.4s, v9.4s, v5.s[1]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "fmla v30.4s, v9.4s, v6.s[1]\n"
      "fmla v31.4s, v9.4s, v7.s[1]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "fmla v24.4s, v10.4s, v0.s[2]\n"
      "fmla v25.4s, v10.4s, v1.s[2]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "add x12, x12, #0x40\n"
      "fmla v26.4s, v10.4s, v2.s[2]\n"
      "fmla v27.4s, v10.4s, v3.s[2]\n"
      "fmla v28.4s, v10.4s, v4.s[2]\n"
      "fmla v29.4s, v10.4s, v5.s[2]\n"
      "fmla v30.4s, v10.4s, v6.s[2]\n"
      "fmla v31.4s, v10.4s, v7.s[2]\n"
      "fmla v24.4s, v11.4s, v0.s[3]\n"
      "fmla v25.4s, v11.4s, v1.s[3]\n"
      "fmla v26.4s, v11.4s, v2.s[3]\n"
      "fmla v27.4s, v11.4s, v3.s[3]\n"
      "fmla v28.4s, v11.4s, v4.s[3]\n"
      "fmla v29.4s, v11.4s, v5.s[3]\n"
      "fmla v30.4s, v11.4s, v6.s[3]\n"
      "fmla v31.4s, v11.4s, v7.s[3]\n"
      "161:"  // Height 8: Multiply loop: Main loop skip
      "cbz x9, 163f\n"
      "162:"  // Height 8: Multiply loop: Odd block loop
      "ldr s0, [x28], #0x4\n"
      "ldr s23, [x27], #0x4\n"
      "sub x9, x9, #0x1\n"
      "ldr s22, [x26], #0x4\n"
      "ldr s21, [x25], #0x4\n"
      "ldr s20, [x24], #0x4\n"
      "ldr s19, [x23], #0x4\n"
      "ldr s18, [x22], #0x4\n"
      "ldr s17, [x21], #0x4\n"
      "ldr q16, [x12, #0x0]\n"
      "fmla v24.4s, v16.4s, v0.s[0]\n"
      "fmla v25.4s, v16.4s, v23.s[0]\n"
      "add x12, x12, #0x10\n"
      "fmla v26.4s, v16.4s, v22.s[0]\n"
      "fmla v27.4s, v16.4s, v21.s[0]\n"
      "fmla v28.4s, v16.4s, v20.s[0]\n"
      "fmla v29.4s, v16.4s, v19.s[0]\n"
      "fmla v30.4s, v16.4s, v18.s[0]\n"
      "fmla v31.4s, v16.4s, v17.s[0]\n"
      "cbnz x9, 162b\n"
      "163:"  // Height 8: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 156b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x27, x11, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "prfm pstl1keep, [x11, #0x0]\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x21, x22, x20, LSL #2\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "prfm pstl1keep, [x21, #0x0]\n"
      "tbz %x[flags], #1, 164f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1r { v17.4s }, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v16.4s }, [x20]\n"
      "fmin v24.4s, v24.4s, v17.4s\n"
      "fmin v25.4s, v25.4s, v17.4s\n"
      "fmin v26.4s, v26.4s, v17.4s\n"
      "fmin v27.4s, v27.4s, v17.4s\n"
      "fmin v28.4s, v28.4s, v17.4s\n"
      "fmin v29.4s, v29.4s, v17.4s\n"
      "fmin v30.4s, v30.4s, v17.4s\n"
      "fmin v31.4s, v31.4s, v17.4s\n"
      "fmax v24.4s, v24.4s, v16.4s\n"
      "fmax v25.4s, v25.4s, v16.4s\n"
      "fmax v26.4s, v26.4s, v16.4s\n"
      "fmax v27.4s, v27.4s, v16.4s\n"
      "fmax v28.4s, v28.4s, v16.4s\n"
      "fmax v29.4s, v29.4s, v16.4s\n"
      "fmax v30.4s, v30.4s, v16.4s\n"
      "fmax v31.4s, v31.4s, v16.4s\n"
      "164:"  // Height 8: No activation
      "cmp x13, #0x4\n"
      "bge 167f\n"
      "tbz x13, #1, 165f\n"
      "str d24, [x11], #0x8\n"
      "str d25, [x27], #0x8\n"
      "str d26, [x26], #0x8\n"
      "str d27, [x25], #0x8\n"
      "str d28, [x24], #0x8\n"
      "str d29, [x23], #0x8\n"
      "str d30, [x22], #0x8\n"
      "str d31, [x21], #0x8\n"
      "tbz x13, #0, 166f\n"
      "st1 { v24.s }[2], [x11]\n"
      "st1 { v25.s }[2], [x27]\n"
      "st1 { v26.s }[2], [x26]\n"
      "st1 { v27.s }[2], [x25]\n"
      "st1 { v28.s }[2], [x24]\n"
      "st1 { v29.s }[2], [x23]\n"
      "st1 { v30.s }[2], [x22]\n"
      "st1 { v31.s }[2], [x21]\n"
      "b 166f\n"
      "165:"  // Height 8: Partial direct writeback: partial_1_0
      "str s24, [x11, #0x0]\n"
      "str s25, [x27, #0x0]\n"
      "str s26, [x26, #0x0]\n"
      "str s27, [x25, #0x0]\n"
      "str s28, [x24, #0x0]\n"
      "str s29, [x23, #0x0]\n"
      "str s30, [x22, #0x0]\n"
      "str s31, [x21, #0x0]\n"
      "166:"  // Height 8: Partial direct writeback: Done
      "b 168f\n"
      "167:"  // Height 8: Full writeback
      "str q24, [x11, #0x0]\n"
      "add x11, x11, #0x10\n"
      "str q25, [x27, #0x0]\n"
      "str q26, [x26, #0x0]\n"
      "str q27, [x25, #0x0]\n"
      "str q28, [x24, #0x0]\n"
      "str q29, [x23, #0x0]\n"
      "str q30, [x22, #0x0]\n"
      "str q31, [x21, #0x0]\n"
      "168:"  // Height 8: Writeback done
      "subs x13, x13, #0x4\n"
      "bgt 149b\n"
      "subs %x[M], %x[M], #0x8\n"
      "beq 170f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 169f\n"
      "add x21, x21, #0x8\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "169:"  // Update direct input
      "mov x20, #0x20\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "170:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
