/**
 * \file pappsomspp/msrun/private/timsmsrunreader.h
 * \date 05/09/2019
 * \author Olivier Langella
 * \brief MSrun file reader for native Bruker TimsTOF raw data
 */

/*******************************************************************************
 * Copyright (c) 2019 Olivier Langella <Olivier.Langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "timsmsrunreader.h"
#include "../../exception/exceptionnotimplemented.h"
#include "../../exception/exceptioninterrupted.h"
#include <QDebug>

using namespace pappso;

TimsMsRunReader::TimsMsRunReader(MsRunIdCstSPtr &msrun_id_csp)
  : MsRunReader(msrun_id_csp)
{
  initialize();
}

TimsMsRunReader::~TimsMsRunReader()
{
  msp_timsData = nullptr;
}

void
TimsMsRunReader::initialize()
{
  msp_timsData = std::make_shared<TimsData>(mcsp_msRunId.get()->getFileName());

  if(msp_timsData == nullptr)
    {
      throw PappsoException(QObject::tr("ERROR in TimsMsRunReader::initialize "
                                        "msp_timsData is null for MsRunId %1")
                              .arg(mcsp_msRunId.get()->toString()));
    }
}


bool
TimsMsRunReader::accept(const QString &file_name) const
{
  qDebug() << file_name;
  return true;
}


pappso::MassSpectrumSPtr
TimsMsRunReader::massSpectrumSPtr([[maybe_unused]] std::size_t spectrum_index)
{
  throw ExceptionNotImplemented(
    QObject::tr("Not yet implemented in TimsMsRunReader %1.\n").arg(__LINE__));
  return pappso::MassSpectrumSPtr();
}


pappso::MassSpectrumCstSPtr
TimsMsRunReader::massSpectrumCstSPtr(std::size_t spectrum_index)
{
  return msp_timsData->getMassSpectrumCstSPtrByRawIndex(spectrum_index);
}


QualifiedMassSpectrum
TimsMsRunReader::qualifiedMassSpectrum(std::size_t spectrum_index,
                                       bool want_binary_data) const
{

  QualifiedMassSpectrum mass_spectrum;

  msp_timsData->getQualifiedMassSpectrumByRawIndex(
    getMsRunId(), mass_spectrum, spectrum_index, want_binary_data);
  return mass_spectrum;
}


void
TimsMsRunReader::readSpectrumCollection(
  SpectrumCollectionHandlerInterface &handler)
{
  readSpectrumCollectionByMsLevel(handler, 0);
}

void
TimsMsRunReader::readSpectrumCollection2(
  const MsRunReadConfig &config, SpectrumCollectionHandlerInterface &handler)
{

  qDebug().noquote() << "Reading the spectrum collection with this "
                        "specific configuration:"
                     << config.toString();

  std::vector<std::size_t> subset_of_tims_frame_ids;

  bool asked_ion_mobility_scan_num_range = false;

  quint32 mobility_scan_num_range_begin =
    std::numeric_limits<quint32>::quiet_NaN();
  quint32 mobility_scan_num_range_end =
    std::numeric_limits<quint32>::quiet_NaN();
  quint32 mobility_scan_num_range_width =
    std::numeric_limits<quint32>::quiet_NaN();

  double mobility_one_over_k0_range_begin =
    std::numeric_limits<double>::quiet_NaN();
  double mobility_one_over_k0_range_end =
    std::numeric_limits<double>::quiet_NaN();

  if(!config
        .getParameterValue(
          MsRunReadConfigParameter::TimsFrameIonMobScanIndexBegin)
        .isNull() &&
     !config
        .getParameterValue(
          MsRunReadConfigParameter::TimsFrameIonMobScanIndexEnd)
        .isNull())
    {
      mobility_scan_num_range_begin =
        config
          .getParameterValue(
            MsRunReadConfigParameter::TimsFrameIonMobScanIndexBegin)
          .toUInt();
      mobility_scan_num_range_end =
        config
          .getParameterValue(
            MsRunReadConfigParameter::TimsFrameIonMobScanIndexEnd)
          .toUInt();

      // We need the range width below.
      mobility_scan_num_range_width =
        mobility_scan_num_range_end + 1 - mobility_scan_num_range_begin;

      asked_ion_mobility_scan_num_range = true;

      // Be sure to check in the frames loop below that the user might
      // have asked for an ion mobility range but on the basis of the 1/K0 unit.
    }

  const std::vector<FrameIdDescr> &frame_id_descr_list =
    msp_timsData->getFrameIdDescrList();

  // Just for the feedback to the user.
  std::size_t scan_count = 0;

  for(auto const &frame_record : msp_timsData->getTimsFrameRecordList())
    {
      if(handler.shouldStop())
        {
          // qDebug() << "The operation was cancelled. Breaking the loop.";
          throw ExceptionInterrupted(
            QObject::tr("Reading timsTOF data cancelled by the user."));
        }

      if(frame_record.frame_id == 0)
        continue;

      if(!config.acceptRetentionTimeInSeconds(frame_record.frame_time))
        continue;

      std::size_t ms_level = 2;
      if(frame_record.msms_type == 0)
        ms_level = 1;

      if(!config.acceptMsLevel(ms_level))
        continue;

      subset_of_tims_frame_ids.push_back(frame_record.frame_id);

      if(mobility_scan_num_range_width)
        {
          scan_count += mobility_scan_num_range_width;
        }
      else
        {
          scan_count += frame_id_descr_list[frame_record.frame_id].m_size;
        }
    }

  // At this point, we have a subset of frame records.
  std::size_t frame_count = subset_of_tims_frame_ids.size();
  qDebug() << "The number of retained RT range- and MS level-matching frames : "
           << frame_count;

  // Inform the handler of the spectrum list so that it can handle feedback to
  // the user.
  handler.spectrumListHasSize(scan_count);

  // Check for m/z range selection
  double mz_range_begin = -1;
  double mz_range_end   = -1;

  if(!config.getParameterValue(MsRunReadConfigParameter::MzRangeBegin)
        .isNull() &&
     !config.getParameterValue(MsRunReadConfigParameter::MzRangeEnd).isNull())
    {
      mz_range_begin =
        config.getParameterValue(MsRunReadConfigParameter::MzRangeBegin)
          .toDouble();

      mz_range_end =
        config.getParameterValue(MsRunReadConfigParameter::MzRangeEnd)
          .toDouble();

      // qDebug() << "The m/z range asked is: " << mz_range_begin
      //          << "--" << mz_range_end;
    }

  // Check for m/z resolution downgrading (mz bins merge)
  // The idea is that we merge a number of mz indices into a single index,
  // which is essentially an increase of the m/z bin size, and therefore
  // a reduction of the resolution/definition of the mass spectrum.
  std::size_t mz_index_merge_window = 0;
  if(!config
        .getParameterValue(
          MsRunReadConfigParameter::TimsFrameMzIndexMergeWindow)
        .isNull())
    {
      mz_index_merge_window =
        config
          .getParameterValue(
            MsRunReadConfigParameter::TimsFrameMzIndexMergeWindow)
          .toUInt();

      // qDebug() << "mz_index_merge_window=" << mz_index_merge_window;
    }

  std::size_t number_of_mobility_scans_set_as_qualified_mass_spectra = 0;

  for(std::size_t tims_frame_id : subset_of_tims_frame_ids)
    {
      if(handler.shouldStop())
        {
          // qDebug() << "The operation was cancelled. Breaking the loop.";
          throw ExceptionInterrupted(
            QObject::tr("Reading timsTOF data cancelled by the user."));
        }

      // qDebug() << "tims_frame_id=" << tims_frame_id;

      const FrameIdDescr &current_frame_record =
        frame_id_descr_list[tims_frame_id];

      TimsFrameCstSPtr tims_frame_csp =
        msp_timsData->getTimsFrameCstSPtrCached(tims_frame_id);

      // If the user wants to select 1/Ko values in a given range, we need to
      // compute the ion mobility scan value starting from that 1/Ko value in
      // *each* frame. Note that the computed mobility_scan_num_begin and
      // mobility_scan_num_end would override thoses possibly set with
      // TimsFramesMsRunReader_mobility_index_begin/end above.

      if(!config
            .getParameterValue(
              MsRunReadConfigParameter::TimsFrameIonMobOneOverK0Begin)
            .isNull() &&
         !config
            .getParameterValue(
              MsRunReadConfigParameter::TimsFrameIonMobOneOverK0End)
            .isNull())
        {
          mobility_one_over_k0_range_begin =
            config
              .getParameterValue(
                MsRunReadConfigParameter::TimsFrameIonMobOneOverK0Begin)
              .toDouble();

          mobility_one_over_k0_range_end =
            config
              .getParameterValue(
                MsRunReadConfigParameter::TimsFrameIonMobOneOverK0End)
              .toDouble();

          mobility_scan_num_range_begin =
            tims_frame_csp.get()->getScanNumFromOneOverK0(
              mobility_one_over_k0_range_begin);

          mobility_scan_num_range_end =
            tims_frame_csp.get()->getScanNumFromOneOverK0(
              mobility_one_over_k0_range_end);

          asked_ion_mobility_scan_num_range = true;
        }

      // Now that we know if the user has asked for an ion mobility range,
      // either using scan indices or 1/K0 values, we need to double check the
      // range borders.

      quint32 count_of_mobility_scans = tims_frame_csp->getTotalNumberOfScans();

      if(asked_ion_mobility_scan_num_range)
        {
          if(mobility_scan_num_range_end > (count_of_mobility_scans - 1))
            {
              mobility_scan_num_range_end = count_of_mobility_scans - 1;
            }
        }
      else
        {
          mobility_scan_num_range_begin = 0;
          mobility_scan_num_range_end   = count_of_mobility_scans - 1;
        }

      // Now, with or without the peak list, we have to craft a qualified mass
      // spectrum that will hold all the data about the data in it.
      QualifiedMassSpectrum mass_spectrum;

      MassSpectrumId spectrum_id;

      spectrum_id.setSpectrumIndex(tims_frame_id);
      spectrum_id.setMsRunId(getMsRunId());

      mass_spectrum.setMassSpectrumId(spectrum_id);

      // We want to document the retention time!
      mass_spectrum.setRtInSeconds(tims_frame_csp.get()->getTime());

      // We do want to document the ms level of the spectrum and possibly
      // the precursor's m/z and charge.
      unsigned int frame_ms_level = tims_frame_csp.get()->getMsLevel();
      mass_spectrum.setMsLevel(frame_ms_level);

      // The scan index is the index of the scan in the *whole* mass data file,
      // it is a sequential number of scans over all the frames.
      std::size_t scan_index = current_frame_record.m_cumulSize -
                               current_frame_record.m_size +
                               mobility_scan_num_range_begin;

      for(quint32 iter_scan_index = mobility_scan_num_range_begin;
          iter_scan_index <= mobility_scan_num_range_end;
          iter_scan_index++)
        {
          mass_spectrum.getMassSpectrumId().setSpectrumIndex(scan_index);

          mass_spectrum.getMassSpectrumId().setNativeId(
            QString("frame_id=%1 scan_index=%2 global_scan_index=%3")
              .arg(tims_frame_id)
              .arg(iter_scan_index)
              .arg(scan_index));

          // qDebug() << "iter:" << iter;

          if(config.needPeakList())
            {
              quint32 mz_minimum_index_out = 0;
              quint32 mz_maximum_index_out = 0;

              auto raw_trace =
                tims_frame_csp.get()->getMobilityScan(iter_scan_index,
                                                      mz_index_merge_window,
                                                      mz_range_begin,
                                                      mz_range_end,
                                                      mz_minimum_index_out,
                                                      mz_maximum_index_out);

              // qDebug() << "Ion mobility scan's raw trace size:"
              //          << raw_trace.size();

              mass_spectrum.setEmptyMassSpectrum(false);

              mass_spectrum.setParameterValue(
                QualifiedMassSpectrumParameter::TimsFrameMzIndexBegin,
                mz_minimum_index_out);
              mass_spectrum.setParameterValue(
                QualifiedMassSpectrumParameter::TimsFrameMzIndexEnd,
                mz_maximum_index_out);

              // qDebug() << "Scan's mz_minimum_index:" << mz_minimum_index_out
              //          << "and mz_maximum_index:" << mz_maximum_index_out;

              // Arrival time
              mass_spectrum.setDtInMilliSeconds(
                tims_frame_csp.get()->getDriftTime(iter_scan_index));
              // 1/K0
              mass_spectrum.setParameterValue(
                QualifiedMassSpectrumParameter::TimsIonMobScanOneOverK0,
                tims_frame_csp.get()->getOneOverK0Transformation(
                  iter_scan_index));

              qDebug();
              mass_spectrum.setMassSpectrumSPtr(
                std::make_shared<MassSpectrum>(raw_trace));

              qDebug();
            }
          else
            {
              mass_spectrum.setEmptyMassSpectrum(true);
            }

          // qDebug() << "mz_index_merge_window=" << mz_index_merge_window;
          handler.setQualifiedMassSpectrum(mass_spectrum);
          ++number_of_mobility_scans_set_as_qualified_mass_spectra;
          scan_index++;
        }
    }

  qDebug() << "Total number of loaded mass spectra:"
           << number_of_mobility_scans_set_as_qualified_mass_spectra;
}

void
TimsMsRunReader::readSpectrumCollectionByMsLevel(
  SpectrumCollectionHandlerInterface &handler, unsigned int ms_level)
{

  qDebug();

  try
    {

      msp_timsData.get()->rawReaderSpectrumCollectionByMsLevel(
        getMsRunId(), handler, ms_level);
    }

  catch(ExceptionInterrupted &)
    {
      qDebug() << "Reading of MS data interrupted by the user.";
    }

  // Now let the loading handler know that the loading of the data has ended.
  // The handler might need this "signal" to perform additional tasks or to
  // cleanup cruft.

  // qDebug() << "Loading ended";
  handler.loadingEnded();
}


std::size_t
TimsMsRunReader::spectrumListSize() const
{
  return msp_timsData->getTotalNumberOfScans();
}


bool
TimsMsRunReader::hasScanNumbers() const
{
  return false;
}


bool
TimsMsRunReader::releaseDevice()
{
  msp_timsData = nullptr;
  return true;
}

bool
TimsMsRunReader::acquireDevice()
{
  if(msp_timsData == nullptr)
    {
      initialize();
    }
  return true;
}


XicCoordSPtr
TimsMsRunReader::newXicCoordSPtrFromSpectrumIndex(std::size_t spectrum_index
                                                  [[maybe_unused]],
                                                  pappso::PrecisionPtr precision
                                                  [[maybe_unused]]) const
{
  throw ExceptionNotImplemented(QObject::tr("Not implemented %1 %2 %3")
                                  .arg(__FILE__)
                                  .arg(__FUNCTION__)
                                  .arg(__LINE__));
}

XicCoordSPtr
TimsMsRunReader::newXicCoordSPtrFromQualifiedMassSpectrum(
  const pappso::QualifiedMassSpectrum &mass_spectrum [[maybe_unused]],
  pappso::PrecisionPtr precision [[maybe_unused]]) const
{
  throw ExceptionNotImplemented(QObject::tr("Not implemented %1 %2 %3")
                                  .arg(__FILE__)
                                  .arg(__FUNCTION__)
                                  .arg(__LINE__));
}

TimsDataSp
TimsMsRunReader::getTimsDataSPtr()
{
  acquireDevice();
  return msp_timsData;
}


Trace
TimsMsRunReader::getTicChromatogram()
{
  // Use the Sqlite database to fetch the total ion current chromatogram (TIC
  // chromatogram).

  acquireDevice();

  // The time unit here is seconds, not minutes!!!
  return msp_timsData->getTicChromatogram();
}
