#!/usr/bin/env python3
"""
Example of simulating a rough sample with a
tanh and Nevot-Croce roughness model using BornAgain.

"""
from matplotlib import pyplot as plt
import bornagain as ba
from bornagain import angstrom, ba_plot as bp, deg


def get_sample(roughness_model):
    # create materials
    vacuum = ba.MaterialBySLD("Vacuum", 0, 0)
    material_ti = ba.MaterialBySLD("Ti", -1.9493e-06, 0)
    material_ni = ba.MaterialBySLD("Ni", 9.4245e-06, 0)
    material_substrate = ba.MaterialBySLD("SiSubstrate", 2.0704e-06, 0)

    # create layers
    ambient_layer = ba.Layer(vacuum)
    ti_layer = ba.Layer(material_ti, 30*angstrom)
    ni_layer = ba.Layer(material_ni, 70*angstrom)
    substrate_layer = ba.Layer(material_substrate)

    # Roughness
    roughness = ba.LayerRoughness(10*angstrom, roughness_model)

    # create sample
    sample = ba.MultiLayer()
    sample.addLayer(ambient_layer)
    for _ in range(10):
        sample.addLayerWithTopRoughness(ti_layer, roughness)
        sample.addLayerWithTopRoughness(ni_layer, roughness)
    sample.addLayerWithTopRoughness(substrate_layer, roughness)

    return sample


def get_simulation(sample):
    n = <%= sm ? 50 : 500 %>
    scan = ba.AlphaScan(n, 2*deg/n, 2*deg)
    scan.setWavelength(1.54*angstrom)

    return ba.SpecularSimulation(scan, sample)


def simulate(roughness_model, title):
    """
    Runs simulation and returns its result.
    """
    sample = get_sample(roughness_model)
    simulation = get_simulation(sample)
    result = simulation.simulate()
    result.setTitle(title)
    return result


if __name__ == '__main__':
    results = [
        simulate(ba.ErfRoughness(), "Névot-Croce"),
        simulate(ba.TanhRoughness(), "Tanh"),
    ]

    bp.plot_multicurve(results)
    bp.show_or_export()
