//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      PyCore/Embed/PyInterpreter.h
//! @brief     Declares functions to expose Python-interpreter functionality to C++.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_PYTHON
#error this header requires Python support
#endif
#ifndef BORNAGAIN_PYCORE_EMBED_PYINTERPRETER_H
#define BORNAGAIN_PYCORE_EMBED_PYINTERPRETER_H

#include "PyCore/Embed/PyObjectDecl.h"
#include <string>
#include <vector>

class PyObjectPtr;

namespace PyInterpreter {

// Python stable ABI
void initialize();

// Python stable ABI
bool isInitialized();

// Python stable ABI
void finalize();

// Python stable ABI
bool checkError();

// Python stable ABI
void addPythonPath(const std::string& path);

// Python stable ABI
void setPythonPath(const std::string& path);

// Python stable ABI
PyObjectPtr import(const std::string& pymodule_name, const std::string& path = "");
// Python stable ABI
void DecRef(PyObject* py_object);

//! Converts String PyObject into string, if possible, or throws exception
std::string pyStrtoString(PyObject* obj);

//! Gets an attribute from a Python module
std::string getStrAttribute(PyObject* py_module, const std::string& attr_name);

//! Calls a Python module function with a given name and a signature `fn() -> str`
void callFunction(PyObject* py_module, const std::string& fn_name);

//! Returns multi-line string representing PATH, PYTHONPATH, sys.path and other info.
std::string runtimeInfo();

//! Returns string representing python stack trace.
std::string stackTrace();

//! Returns the full error description via Python stack trace
std::string errorDescription(const std::string& title = "");

//-- Numpy-related functionality
namespace Numpy {

typedef long int np_size_t; // size type used in Numpy

//! Initializes Numpy
int initialize();

//! Checks if Numpy is initialized
bool isInitialized();

//! Creates an empty Numpy array of type `double` with the given dimensions
PyObjectPtr arrayND(const std::vector<std::size_t>& dimensions);

//! Returns the pointer to the data buffer of a Numpy array of type `double`
double* getDataPtr(PyObject* pyobject_ptr);

PyObject* fromCppVector(const std::vector<std::size_t>& dimensions,
                        const std::vector<double>& cpp_vec);

//! Creates a Numpy 1D-array from a contiguous C-array of type `double`.
// Data will be copied.
PyObjectPtr createArray1DfromC(double* const c_array, const np_size_t size);

//! Creates a Numpy 2D-array from a contiguous C-array of type `double`.
//! Data will be copied.
PyObjectPtr createArray2DfromC(double* const c_array, const np_size_t dims[2]);

//! Creates a `vector<double>` from a Numpy 2D-array (data will be copied).
std::vector<double> createVectorFromArray2D(PyObject* pyobject_ptr);

//! Creates a 2D Numpy array from a contiguous C-array of type `double`.
//! The data is _not_ copied, and the Numpy array will _not_ own its data.
PyObjectPtr CArrayAsNpy2D(double* const c_array, const np_size_t dims[2]);

} // namespace Numpy

// BornAgain-related functionality
namespace BornAgain {

//! Imports BornAgain from given location. If path is empty, tries to rely on PYTHONPATH.
PyObjectPtr import(const std::string& path = "");

//! Imports a 'BornAgain' Python script
PyObjectPtr importScript(const std::string& script, const std::string& path);

//! Returns list of functions defined in the script, given a script filename and
//! a path to import BornAgain library (if empty, relies on `PYTHONPATH`).
std::vector<std::string> listOfFunctions(const std::string& script, const std::string& path);

//! Call a BornAgain-related function and return its returned value
PyObjectPtr callScriptFunction(const std::string& functionName, const std::string& script,
                               const std::string& path);

} // namespace BornAgain

// Fabio-related functionality
namespace Fabio {

//! Imports 'fabio' Python module
PyObjectPtr import();
//! Calls 'fabio.open' on a given file (needs Numpy)
PyObjectPtr open(const std::string& filename, PyObjectPtr& fabio_module);

} // namespace Fabio

} // namespace PyInterpreter

#endif // BORNAGAIN_PYCORE_EMBED_PYINTERPRETER_H
