//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Frame/Fit1DFrame.cpp
//! @brief     Implements class Fit1DFrame.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Frame/Fit1DFrame.h"
#include "Base/Util/Assert.h"
#include "Device/Data/DataUtil.h"
#include "Device/Data/Datafield.h"
#include "GUI/Model/Axis/AmplitudeAxisItem.h"
#include "GUI/Model/Data/Data1DItem.h"
#include "GUI/Model/Data/RangeUtil.h"
#include "GUI/Model/Job/JobsSet.h"
#include "GUI/Model/Project/DataSource.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/Canvas/ProgressCanvas.h"
#include "GUI/View/Canvas/SpecularPlotCanvas.h"
#include "GUI/View/Plotter/PlotStatusLabel.h"
#include "GUI/View/Plotter/SpecularPlot.h"
#include "GUI/View/Setup/AxisPanel.h"
#include "GUI/View/Setup/FrameActions.h"

Fit1DFrame::Fit1DFrame()
    : AnydataFrame(std::make_unique<DataFromJob>())
    , m_data_canvas(new SpecularPlotCanvas)
    , m_diff_canvas(new SpecularPlotCanvas)
    , m_status_label(
          new PlotStatusLabel({m_data_canvas->specularPlot(), m_diff_canvas->specularPlot()}))
    , m_reset_view_action(new QAction(this))
{
    auto* vlayout = new QVBoxLayout;
    vlayout->setContentsMargins(0, 0, 0, 0);
    vlayout->setSpacing(0);

    auto* gridLayout = new QGridLayout;
    gridLayout->setContentsMargins(0, 0, 0, 0);
    gridLayout->setSpacing(0);

    gridLayout->addWidget(m_data_canvas, 0, 0, 1, -1);
    gridLayout->addWidget(m_diff_canvas, 1, 0, 1, 2);
    auto* progress_canvas = new ProgressCanvas;
    gridLayout->addWidget(progress_canvas, 1, 2, 1, 1);

    vlayout->addLayout(gridLayout);
    vlayout->addWidget(m_status_label);

    auto* hlayout = new QHBoxLayout(this);
    hlayout->setContentsMargins(0, 0, 0, 0);
    hlayout->setSpacing(0);
    hlayout->addLayout(vlayout);

    auto* axis_panel = new AxisPanel(dataSource());
    hlayout->addWidget(axis_panel);
    axis_panel->hide();

    // TODO restore when we have a toolbar here
    // QObject::connect(toolbar->actions()->toggle_properties_panel, &QAction::triggered,
    // axis_panel,
    //                 &QWidget::setVisible);

    m_reset_view_action->setText("Center view");
    m_reset_view_action->setIcon(QIcon(":/images/camera-metering-center.svg"));
    m_reset_view_action->setToolTip("Reset View");
    connect(m_reset_view_action, &QAction::triggered, this, &Fit1DFrame::onResetViewAction);

    GUI::Util::Ranges::setCommonRangeY(dataSource()->mainData1DItems());

    connect(gDoc->jobs(), &JobsSet::setChanged, this, &Fit1DFrame::updateFrame);

    updateFrame();
}

Fit1DFrame::~Fit1DFrame() = default;

void Fit1DFrame::onResetViewAction()
{
    ASSERT(dataSource()->simuData1DItem() && dataSource()->diffData1DItem()
           && dataSource()->realData1DItem());
    dataSource()->simuData1DItem()->resetView();
    dataSource()->realData1DItem()->resetView();
    dataSource()->diffData1DItem()->resetView();

    // synchronize data range between simulated and real
    GUI::Util::Ranges::setCommonRangeY(dataSource()->mainData1DItems());
    gDoc->setModified();
}

void Fit1DFrame::updateFrame()
{
    if (dataSource()->simuData1DItem() && dataSource()->realData1DItem()) {
        m_data_canvas->setData1DItems(
            {dataSource()->simuData1DItem(), dataSource()->realData1DItem()});
        m_diff_canvas->setData1DItems({dataSource()->diffData1DItem()});
        // sync X axis view area between simulated and difference plots
        // simu --> diff
        connect(dataSource()->simuData1DItem(), &DataItem::updateOtherPlots,
                dataSource()->diffData1DItem(), &DataItem::checkXranges, Qt::UniqueConnection);
        // diff --> simu
        connect(dataSource()->diffData1DItem(), &DataItem::updateOtherPlots,
                dataSource()->simuData1DItem(), &DataItem::checkXranges, Qt::UniqueConnection);

        // update diff data if simulation data changes
        connect(dataSource()->simuData1DItem(), &Data1DItem::datafieldChanged, this,
                &Fit1DFrame::updateDiffData, Qt::UniqueConnection);
        updateDiffData();
        show();
    } else
        hide();
}

void Fit1DFrame::updateDiffData()
{
    ASSERT(dataSource()->simuData1DItem() && dataSource()->diffData1DItem()
           && dataSource()->realData1DItem());
    if (!dataSource()->simuData1DItem()->c_field() || !dataSource()->realData1DItem()->c_field())
        return;

    dataSource()->diffData1DItem()->setDatafield(DataUtil::relativeDifferenceField(
        *dataSource()->simuData1DItem()->c_field(), *dataSource()->realData1DItem()->c_field()));

    // keep Y axis range up with data range
    double min = dataSource()->diffData1DItem()->yMin();
    double max = dataSource()->diffData1DItem()->yMax();
    if (!dataSource()->diffData1DItem()->axItemY()->isLogScale() || min > 0)
        dataSource()->diffData1DItem()->setYrange(min, max);
}
