#include "Device/Mask/Polygon.h"
#include "Base/Axis/Bin.h"
#include "Tests/GTestWrapper/google_test.h"
#include <memory>

TEST(PolygonTest, SimpleRectangle)
{
    // simple closed rectangle
    std::vector<double> x = {4.0, -4.0, -4.0, 4.0, 4.0};
    std::vector<double> y = {2.0, 2.0, -2.0, -2.0, 2.0};
    Polygon polygon(x, y);
    EXPECT_DOUBLE_EQ(32.0, polygon.getArea());
    EXPECT_TRUE(polygon.contains(0.0, 0.0));
    EXPECT_TRUE(polygon.contains(4.0, 2.0));
    EXPECT_TRUE(polygon.contains(-4.0, -2.0));
    EXPECT_TRUE(polygon.contains(-4.0, -2.0));
    EXPECT_FALSE(polygon.contains(0.0, 2.01));
    EXPECT_FALSE(polygon.contains(4.0, -2.01));

    // unclosed rectangle (should be closed automatically)
    x = {4.0, -4.0, -4.0, 4.0};
    y = {2.0, 2.0, -2.0, -2.0};
    Polygon polygon2(x, y);
    EXPECT_DOUBLE_EQ(32.0, polygon2.getArea());
    EXPECT_TRUE(polygon2.contains(0.0, 0.0));
    EXPECT_TRUE(polygon2.contains(4.0, 2.0));
    EXPECT_TRUE(polygon2.contains(-4.0, -2.0));
    EXPECT_TRUE(polygon2.contains(-4.0, -2.0));
    EXPECT_FALSE(polygon2.contains(0.0, 2.01));
    EXPECT_FALSE(polygon2.contains(4.0, -2.01));
}

//  ************************************************************************************************
//      *   *
//       * *
//        *
//       * *
//      *   *
//  ************************************************************************************************

TEST(PolygonTest, SandWatchShape)
{
    std::vector<double> x = {2.0, -2.0, 2.0, -2.0, 2.0};
    std::vector<double> y = {2.0, 2.0, -2.0, -2.0, 2.0};
    Polygon polygon(x, y);
    //    std::cout << polygon << std::endl;

    // for some reason area calculation doesn't work for boost's polygon of such shape
    // EXPECT_DOUBLE_EQ(8.0, polygon.getArea());

    EXPECT_TRUE(polygon.contains(2.0, 2.0));
    EXPECT_TRUE(polygon.contains(-2.0, 2.0));
    EXPECT_TRUE(polygon.contains(0.0, 0.0));
    EXPECT_TRUE(polygon.contains(0.0, 1.0));

    EXPECT_FALSE(polygon.contains(1.0, 0.0));
    EXPECT_FALSE(polygon.contains(-1.5, 0.5));
}

TEST(PolygonTest, ContainsBin)
{
    // simple closed rectangle
    std::vector<double> x = {4.0, -4.0, -4.0, 4.0, 4.0};
    std::vector<double> y = {2.0, 2.0, -2.0, -2.0, 2.0};
    Polygon polygon(x, y);

    Bin1D binx1 = Bin1D::FromTo(3.5, 4.5);
    Bin1D biny1 = Bin1D::FromTo(1.5, 2.5);
    EXPECT_TRUE(polygon.contains(binx1, biny1));

    Bin1D binx2 = Bin1D::FromTo(3.5, 4.6);
    Bin1D biny2 = Bin1D::FromTo(1.5, 2.6);
    EXPECT_FALSE(polygon.contains(binx2, biny2));
}

TEST(PolygonTest, Clone)
{
    std::vector<double> x = {4.0, -4.0, -4.0, 4.0, 4.0};
    std::vector<double> y = {2.0, 2.0, -2.0, -2.0, 2.0};
    Polygon polygon(x, y);

    std::unique_ptr<Polygon> clone(polygon.clone());
    EXPECT_DOUBLE_EQ(32.0, clone->getArea());
    EXPECT_TRUE(clone->contains(0.0, 0.0));
    EXPECT_TRUE(clone->contains(4.0, 2.0));
    EXPECT_TRUE(clone->contains(-4.0, -2.0));
    EXPECT_TRUE(clone->contains(-4.0, -2.0));
    EXPECT_FALSE(clone->contains(0.0, 2.01));
    EXPECT_FALSE(clone->contains(4.0, -2.01));
}

TEST(PolygonTest, ConstructFrom2DArray)
{
    // simple closed rectangle
    std::vector<std::pair<double, double>> points = {
        {4.0, 2.0}, {-4.0, 2.0}, {-4.0, -2.0}, {4.0, -2.0}, {4.0, 2.0}};

    Polygon polygon(points);
    EXPECT_DOUBLE_EQ(32.0, polygon.getArea());
    EXPECT_TRUE(polygon.contains(0.0, 0.0));
    EXPECT_TRUE(polygon.contains(4.0, 2.0));
    EXPECT_TRUE(polygon.contains(-4.0, -2.0));
    EXPECT_TRUE(polygon.contains(-4.0, -2.0));
    EXPECT_FALSE(polygon.contains(0.0, 2.01));
    EXPECT_FALSE(polygon.contains(4.0, -2.01));
}
