/*
 * This module provides the linear regression functions used by bing.
 */

/* $Id: lin_reg.h,v 1.3 1999/10/11 05:25:19 fgouget Exp $ */

#ifndef _lin_reg_h_
#define _lin_reg_h_

#include "bing_defs.h"

typedef struct {
    /*
     * The resulting equation parameters: y = a * x + b
     * We're only interested in a.
     */
    double a;
    double b;
    double corr;
    
    /* The parameters used to compute a, b and the correlation */
    int nb_samples;
    double sum_x;
    double sum_y;
    double sum_x2;
    double sum_y2;
    double sum_xy;
} linreg_t;

/* (!!) */
extern BINGAPI int BINGPROTO(
    linreg_init,
    (
        linreg_t *linreg
    ));

/* (!!) update
  --------------------------------------------------------------------------
	linreg_compute computes performs the lirnear regression from 
	scratch and thus requires that you provide all the statistic 
	samples. The result is a data structure containing all that is 
	necessary to later update the linear regression results if samples
	are added or removed. Ignores any sample for which y is DBL_MAX.
  ------------+----+-----+--------------------------------------------------
  Parameter   | IN | OUT | Role
  ------------+----+-----+--------------------------------------------------
  nb_samples  | X  |     | The number of samples
  x           | X  |     | The x sample values
  y           | X  |     | The y sample values
  linreg      |    |  X  | The linear regression results
  ------------+----+-----+--------------------------------------------------
  RETURN      |    |  X  | 0 if successful, -1 otherwise
  ------------+----+-----+--------------------------------------------------
*/
extern BINGAPI int BINGPROTO(
    linreg_add_all_samples,
    (
        linreg_t *linreg,
        int nb_samples,
        int* x,
        double* y,
        int update
    ));

/* (!!) update
  --------------------------------------------------------------------------
	linreg_del_sample updates the linear regression data when a sample 
	is removed. Note that repeteadly calling this function will degrade 
	the results precision.
  ------------+----+-----+--------------------------------------------------
  Parameter   | IN | OUT | Role
  ------------+----+-----+--------------------------------------------------
  nb_samples  | X  |     | The number of samples after the sample deletion
              |    |     | If nul then the linreg values are not updated
  x           | X  |     | The x value of the sample to delete
  y           | X  |     | The y value of the sample to delete
  linreg      | X  |  X  | The linear regression results
  ------------+----+-----+--------------------------------------------------
  RETURN      |    |  X  | No return value
  ------------+----+-----+--------------------------------------------------
*/
extern BINGAPI int BINGPROTO(
    linreg_del_sample,
    (
        linreg_t *linreg,
        int x,
        double y,
        int update
    ));

/* (!!) update
  --------------------------------------------------------------------------
	linreg_add_sample updates the linear regression data when a 
	sample is added. Note that repeteadly calling this function will
	degrade the results precision.
  ------------+----+-----+--------------------------------------------------
  Parameter   | IN | OUT | Role
  ------------+----+-----+--------------------------------------------------
  nb_samples  | X  |     | The number of samples after the sample addition
              |    |     | If nul then the linreg values are not updated
  x           | X  |     | The x value of the sample to add
  y           | X  |     | The y value of the sample to add
  linreg      | X  |  X  | The linear regression results
  ------------+----+-----+--------------------------------------------------
  RETURN      |    |  X  | No return value
  ------------+----+-----+--------------------------------------------------
*/
extern BINGAPI int BINGPROTO(
    linreg_add_sample,
    (
        linreg_t *linreg,
        int x,
        double y,
        int update
    ));

extern BINGAPI void BINGPROTO(
    linreg_update,
    (
        linreg_t *linreg
    ));

#endif /* end of file */
